//	TorusGamesTwoFingerGestureRecognizer.m
//
//	© 2021 by Jeff Weeks
//	See TermsOfUse.txt

#import "TorusGamesTwoFingerGestureRecognizer.h"
#import <UIKit/UIGestureRecognizerSubclass.h>


//	When the user begins a two-finger gesture, the first touch
//	might move slightly before the second touch begins.
//	Moved to UIGestureRecognizerStateFailed if and only if
//
//		more than SECOND_TOUCH_TIME_TOLERANCE seconds elapse
//	or
//		the first finger moves a distance greater than SECOND_TOUCH_DISTANCE_TOLERANCE
//
//	before a second touch begins.
#define SECOND_TOUCH_TIME_TOLERANCE				0.05	//	seconds
#define	SECOND_TOUCH_DISTANCE_TOLERANCE			4.0		//	points
#define SECOND_TOUCH_DISTANCE_TOLERANCE_SQUARED	(SECOND_TOUCH_DISTANCE_TOLERANCE * SECOND_TOUCH_DISTANCE_TOLERANCE)


//	Privately-declared methods
@interface TorusGamesTwoFingerGestureRecognizer()
@end


@implementation TorusGamesTwoFingerGestureRecognizer
{
	//	For use in deciding when to stop waiting for a second touch.
	NSTimeInterval	itsFirstTouchBeganTime;		//	in seconds
	CGPoint			itsFirstTouchBeganPoint;	//	in points
}


- (id)initWithTarget:(id)target action:(SEL)action
{
	self = [super initWithTarget:target action:action];
	if (self != nil)
	{
		itsFirstTouchBeganTime	= 0.0;
		itsFirstTouchBeganPoint	= CGPointZero;
	}
	return self;
}


- (void)reset
{
	[super reset];
	
	itsFirstTouchBeganTime	= 0.0;
	itsFirstTouchBeganPoint	= CGPointZero;
}

- (void)touchesBegan:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	[super touchesBegan:touches withEvent:event];
	
	switch ([self state])
	{
		case UIGestureRecognizerStatePossible:

			//	"touches"
			//		is the set of newly begun touches.
			//
			//	[event touchesForGestureRecognizer:self]
			//		is the set of all current touches.
			//	
			switch ([[event touchesForGestureRecognizer:self] count])
			{
				case 1:
					itsFirstTouchBeganTime	= [event timestamp];
					itsFirstTouchBeganPoint	= [self locationOfTouch:0 inView:[self view]];
					break;
				
				case 2:
					[self setState:UIGestureRecognizerStateBegan];
					break;
				
				default:
					[self setState:UIGestureRecognizerStateFailed];
					break;
			}

			break;
		
		case UIGestureRecognizerStateBegan:
		case UIGestureRecognizerStateChanged:
			[self setState:UIGestureRecognizerStateEnded];
			break;

		case UIGestureRecognizerStateEnded:
		case UIGestureRecognizerStateCancelled:
		case UIGestureRecognizerStateFailed:
			break;
	}
}

- (void)touchesMoved:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	NSTimeInterval	theElapsedTime;	//	time since first (and only) touch began
	CGPoint			thePoint,
					theDifference;
	double			theDistanceSq;	//	squared distance that first (and only) touch has moved

	[super touchesMoved:touches withEvent:event];
	
	switch ([self state])
	{
		case UIGestureRecognizerStatePossible:	//	exactly 1 touch is active

			theElapsedTime	= [event timestamp] - itsFirstTouchBeganTime;

			thePoint		= [self locationOfTouch:0 inView:[self view]];
			theDifference.x	= thePoint.x - itsFirstTouchBeganPoint.x;
			theDifference.y	= thePoint.y - itsFirstTouchBeganPoint.y;
			theDistanceSq	= theDifference.x * theDifference.x
							+ theDifference.y * theDifference.y;

			if (theElapsedTime > SECOND_TOUCH_TIME_TOLERANCE
			 || theDistanceSq  > SECOND_TOUCH_DISTANCE_TOLERANCE_SQUARED)
			{
				[self setState:UIGestureRecognizerStateFailed];
			}

			break;
		
		case UIGestureRecognizerStateBegan:
			[self setState:UIGestureRecognizerStateChanged];
			break;
		
		case UIGestureRecognizerStateChanged:
		case UIGestureRecognizerStateEnded:
		case UIGestureRecognizerStateCancelled:
		case UIGestureRecognizerStateFailed:
			break;
	}
}

- (void)touchesEnded:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	[super touchesEnded:touches withEvent:event];
	
	switch ([self state])
	{
		case UIGestureRecognizerStatePossible:
			[self setState:UIGestureRecognizerStateFailed];
			break;
		
		case UIGestureRecognizerStateBegan:
		case UIGestureRecognizerStateChanged:
			[self setState:UIGestureRecognizerStateEnded];
			break;
		
		case UIGestureRecognizerStateEnded:
		case UIGestureRecognizerStateCancelled:
		case UIGestureRecognizerStateFailed:
			break;
	}
}

- (void)touchesCancelled:(NSSet<UITouch *> *)touches withEvent:(UIEvent *)event
{
	[super touchesEnded:touches withEvent:event];

	switch ([self state])
	{
		case UIGestureRecognizerStatePossible:
			[self setState:UIGestureRecognizerStateFailed];
			break;
		
		case UIGestureRecognizerStateBegan:
		case UIGestureRecognizerStateChanged:
			[self setState:UIGestureRecognizerStateEnded];
			break;
		
		case UIGestureRecognizerStateEnded:
		case UIGestureRecognizerStateCancelled:
		case UIGestureRecognizerStateFailed:
			break;
	}
}


@end
